/**
 * @file
 * This script contains two event listeners that are executed when the DOM
 *   content is loaded. The first listener handles the click event on elements
 *   with the class modal-link, while the second listener handles the click
 *   event on elements with the class mapLink. In the first listener, when a
 *   modal-link element is clicked, it retrieves the corresponding modal ID,
 *   toggles the visibility by adding or removing the classes modal-visible and
 *   modal-hidden, and calculates the position for the modal. In the second
 *   listener, when a mapLink element is clicked, it retrieves the map
 *   container, toggles the visibility by adding or removing the classes
 *   map-visible and map-hidden, and updates the innerHTML of the link based on
 *   the visibility state.
 */
(function ($, Drupal) {
  'use strict';

  Drupal.behaviors.performances = {

    attach: function (context) {
      once('modal-links', '.modal-link', context).forEach(link => {
        link.addEventListener('click', function (event) {
          event.preventDefault();
          let modalId = this.getAttribute('data-modal-id');
          modalId = 'id-' + modalId;
          let modal = document.getElementById(modalId);
          let isVisible = modal.classList.contains('modal-visible');
          if (isVisible) {
            modal.classList.remove('modal-visible');
            modal.classList.add('modal-hidden');
          }
          else {
            modal.classList.remove('modal-hidden');
            modal.classList.add('modal-visible');
          }

          let rect = this.getBoundingClientRect();
          let scrollTop = window.pageYOffset || document.documentElement.scrollTop;
          let position;

          if (window.innerWidth < 992) {
            position = rect.top + scrollTop - 220;
          }
          else {
            position = rect.top + scrollTop - 190;
          }

          modal.style.top = position + 'px';
        });
      });

      once('carousel', '#event-performances-carousel', context).forEach(element => {
        let carousel = $('#event-performances-carousel');
        let containerSize = $('.carousel-item').innerHeight();
        $('.carousel-item').css('min-height', containerSize)
        $(context).ready(function() {
          hidden_prev_button();
        });
        carousel.on('slid.bs.carousel', event => {
          hidden_next_button();
          hidden_prev_button();
        });
      });

      function hidden_next_button() {
        if ($('.carousel-inner .carousel-item:last').hasClass('active')) {
          $('.carousel-control-next').addClass('d-none');
        } else {
          $('.carousel-control-next').removeClass('d-none');
        }
      }

      function hidden_prev_button() {
        if ($('.carousel-inner .carousel-item:first').hasClass('active')) {
          $('.carousel-control-prev').addClass('d-none');
        } else {
          $('.carousel-control-prev').removeClass('d-none');
        }
      }

    },
  };

})(jQuery, Drupal);
